import "./index.scss"

import { registerPlugin } from '@wordpress/plugins';
import { PluginDocumentSettingPanel } from '@wordpress/edit-post';
import { Button, Icon, PanelRow, SelectControl, RadioControl, TextareaControl} from '@wordpress/components';
import { useState } from '@wordpress/element';
import apiFetch from '@wordpress/api-fetch';

const __ = wp.i18n.__;


// Block editor commands:
// https://developer.wordpress.org/block-editor/reference-guides/packages/packages-block-editor/#commands
// wp.data.select( 'core/block-editor' ).getSelectedBlock() - get one selected (or with cursor inside) block object
// wp.data.select( 'core/block-editor' ).getMultiSelectedBlocks() - get selected blocks objects as array
// wp.data.dispatch( 'core/block-editor' ).insertBlock( block, index, rootClientId, updateSelection )
// wp.data.dispatch( 'core/block-editor' ).replaceBlock( clientId, block )
// wp.data.dispatch( 'core/block-editor' ).removeBlock( clientId, selectParent )


const TrxAddonsAiHelperSettingPanel = () => {
	const [busy, setBusy] = useState( false );

	const aiHelperCommandList = TRX_ADDONS_STORAGE[ 'ai_helper_list_commands' ];
	const [aiHelperCommand, setAiHelperCommand] = useState( 'write_blog' ); 

	const aiHelperTextToneList = TRX_ADDONS_STORAGE[ 'ai_helper_list_text_tones' ];
	const [aiHelperTextTone, setAiHelperTextTone] = useState( 'normal' );
	const aiHelperTextToneOptions = [];
	Object.keys( aiHelperTextToneList ).forEach( function( key ) {
		aiHelperTextToneOptions.push( { label: aiHelperTextToneList[key], value: key } );
	} );

	const aiHelperTextLanguageList = TRX_ADDONS_STORAGE[ 'ai_helper_list_text_languages' ];
	const [aiHelperTextLanguage, setAiHelperTextLanguage] = useState( 'english' );
	const aiHelperTextLanguageOptions = [];
	Object.keys( aiHelperTextLanguageList ).forEach( function( key ) {
		aiHelperTextLanguageOptions.push( { label: aiHelperTextLanguageList[key], value: key } );
	} );

	const aiHelperBaseList = TRX_ADDONS_STORAGE[ 'ai_helper_list_bases' ];
	const [aiHelperBasedOn, setAiHelperBasedOn] = useState( 'prompt' );
	const aiHelperBaseOptions = [];
	Object.keys( aiHelperBaseList ).forEach( function( key ) {
		// Hide an option 'Post title' if the command is not 'write_...' and not 'translate'
		if ( key == 'title' && aiHelperCommand.indexOf( 'write_' ) < 0 && aiHelperCommand != 'process_translate' ) return;
		// Hide an option 'Post excerpt' if the command is not 'translate'
		if ( key == 'excerpt' && aiHelperCommand != 'process_translate' ) return;
		// Hide options 'Selected text' and 'Post content' if the command is 'write_...'
		if ( ['selected', 'content'].indexOf( key ) >= 0 && aiHelperCommand.indexOf( 'write_' ) === 0 ) return;
		// Add option to the list
		aiHelperBaseOptions.push( { label: aiHelperBaseList[key], value: key } );
	} );

	const [aiHelperPrompt, setAiHelperPrompt] = useState( aiHelperBasedOn == 'prompt' ? aiHelperCommandList[aiHelperCommand].prompt : '' ); 
	const [aiHelperHint, setAiHelperHint] = useState( '' );

	const getSelectedText = () => {
		var text = '';
		if ( window.getSelection ) {
			text = window.getSelection().toString();
		} else if ( document.selection && document.selection.type != "Control" ) {
			text = document.selection.createRange().text;
		}
		return text;
	}

	const getAiResponse = () => {

		if ( busy ) return;
		setBusy( true );

		async function go() {
	
			const response = await apiFetch( {
				path: '/ai-helper/v1/get-response',
				method: 'POST',
				data: {
					content: aiHelperBasedOn == 'content'
								? wp.data.select( 'core/editor' ).getEditedPostAttribute( 'content' )
								: ( aiHelperBasedOn == 'title'
									? wp.data.select( 'core/editor' ).getEditedPostAttribute( 'title' )
									: ( aiHelperBasedOn == 'excerpt'
										? wp.data.select( 'core/editor' ).getEditedPostAttribute( 'excerpt' )
										: ( aiHelperBasedOn == 'selected'
											? getSelectedText() || wp.data.select( 'core/block-editor' ).getSelectedBlock().attributes.content
											: ''
											)
										)
									),
					command: aiHelperCommand,
					base_on: aiHelperBasedOn,
					prompt:  aiHelperPrompt,
					hint:	 aiHelperHint,
					text_tone: aiHelperTextTone,
					text_language: aiHelperTextLanguage
				}
			} );
			
			setBusy( false );

			if ( response && response.hasOwnProperty( 'data' ) && response.hasOwnProperty( 'error' ) ) {
				if ( response.data ) {
					var data = response.data;
					var html = '';
					var variations = aiHelperCommandList[aiHelperCommand].variations;
					if ( variations ) {
						for (var i = 0; i < data.length; i++ ) {
							html += '<label class="trx_addons_ai_helper_response__variant"><input name="trx_addons_ai_helper_response__variant" type="radio" value="' + data[i] + '">' + data[i] + '</label>';
						}
					} else {
						html = '<textarea class="trx_addons_ai_helper_response__content" name="trx_addons_ai_helper_response__content" rows="15">' + trx_addons_esc_html( data[0] ) + '</textarea>';
					}
					var buttons = [ TRX_ADDONS_STORAGE['msg_ai_helper_bt_caption_replace'], TRX_ADDONS_STORAGE['msg_ai_helper_bt_caption_append'], TRX_ADDONS_STORAGE['msg_caption_cancel'] ];
					if ( ['process_title', 'process_excerpt'].indexOf( aiHelperCommand ) >= 0 ) {
						buttons = [ TRX_ADDONS_STORAGE['msg_caption_apply'], TRX_ADDONS_STORAGE['msg_caption_cancel'] ];
					} else if ( aiHelperCommand == 'process_heading' ) {
						buttons = [ TRX_ADDONS_STORAGE['msg_ai_helper_bt_caption_prepend'], TRX_ADDONS_STORAGE['msg_caption_cancel'] ];
					}
					trx_addons_msgbox_dialog(
						'<form name="trx_addons_ai_helper_response" class="trx_addons_ai_helper_response">' + html + '</form>',
						TRX_ADDONS_STORAGE[ 'msg_ai_helper_response' + ( variations ? '_variations' : '') ],
						null,
						function( btn, box ) {
							if ( btn < buttons.length ) {
								var value = variations
												? box.find('input[name="trx_addons_ai_helper_response__variant"]:checked').val()
												: box.find('textarea').text();
								// Set the selected variation to the post title field
								if ( aiHelperCommand == 'process_title' || ( aiHelperCommand == 'process_translate' && aiHelperBasedOn == 'title' ) ) {
									wp.data.dispatch( 'core/editor' ).editPost( { title: trx_addons_clear_tags( value ) } );

								// Set the selected variation to the post excerpt field
								} else if ( aiHelperCommand == 'process_excerpt' || ( aiHelperCommand == 'process_translate' && aiHelperBasedOn == 'excerpt' ) ) {
									wp.data.dispatch( 'core/editor' ).editPost( { excerpt: trx_addons_clear_tags( value ) } );

								// Set the response to the post content field
								} else {
									// Convert the response to the blocks
									// if ( value.indexOf( '<!-- wp:' ) == -1 ) {
									// 	value = '<!-- wp:paragraph --><p>' + value + '</p><!-- /wp:paragraph -->';
									// }
									// Convert the response to the heading
									if ( aiHelperCommand == 'process_heading' ) {
										value = '<h3>' + value + '</h3>';
									}
									value = wp.blocks.serialize( wp.blocks.rawHandler( { HTML: value } ) );
									// Parse the response to blocks
									var blocks = wp.blocks.parse( value );
									// Put the response instead / after selected blocks
									if ( aiHelperBasedOn == 'selected' ) {
										var selectedBlockIds = wp.data.select( 'core/block-editor' ).getSelectedBlockCount() > 1
																? wp.data.select( 'core/block-editor' ).getMultiSelectedBlockClientIds()
																: [ wp.data.select( 'core/block-editor' ).getSelectedBlockClientId() ];
										if ( btn == 1 ) {
											if ( aiHelperCommand == 'process_heading' ) {
												// Prepend the response to the selected blocks
												wp.data.dispatch( 'core/block-editor' ).insertBlocks( blocks, wp.data.select( 'core/block-editor' ).getBlockIndex( selectedBlockIds[0], wp.data.select( 'core/block-editor' ).getSelectedBlockClientId() ) );
											} else {
												// Replace the selected blocks with the response
												wp.data.dispatch( 'core/block-editor' ).replaceBlocks( selectedBlockIds, blocks );
											}
										} else {
											// Put the response after the selected blocks
											wp.data.dispatch( 'core/block-editor' ).insertBlocks( blocks, wp.data.select( 'core/block-editor' ).getBlockIndex( selectedBlockIds[ selectedBlockIds.length - 1 ], wp.data.select( 'core/block-editor' ).getSelectedBlockClientId() ) + 1 );
										}

									// Put the response instead / after the content
									} else {
										//wp.data.dispatch( 'core/editor' ).editPost( { content: value } );
										if ( btn == 1 ) {
											if ( aiHelperCommand == 'process_heading' ) {
												// Prepend the response to the content
												wp.data.dispatch( 'core/editor' ).insertBlocks( blocks, 0 );
											} else {
												// Replace the content with the response
												wp.data.dispatch( 'core/editor' ).resetBlocks( blocks );
											}
										} else {
											// Put the response after the content
											wp.data.dispatch( 'core/editor' ).insertBlocks( blocks );
										}
									}
								}
							}
						},
						buttons,
						'wpforms'
					);
				} else {
					trx_addons_msgbox_warning( response.error ? response.error : TRX_ADDONS_STORAGE['msg_ai_helper_error'], TRX_ADDONS_STORAGE['msg_ai_helper_error'] );
				}
			} else {
				trx_addons_msgbox_warning( TRX_ADDONS_STORAGE['msg_ai_helper_error'] );
			}
		}
	
		go();
	};

	return (
		<PluginDocumentSettingPanel
			name="trx-addons-ai-helper-panel"
			title={ __( "TRX Addons AI Helper" ) }
			className="trx-addons-ai-helper-panel"
		>
			<PanelRow>
				<SelectControl
					label={ __( 'AI Command:' ) }
					value={ aiHelperCommand }
					onChange={ ( selection ) => {
						setAiHelperCommand( selection );
						if ( selection.indexOf('write_') === 0 && ['prompt', 'title'].indexOf( aiHelperBasedOn ) == -1 ) {
							setAiHelperBasedOn( 'prompt' );
							setAiHelperPrompt( aiHelperCommandList[selection].prompt );
						} else {
							setAiHelperPrompt( aiHelperBasedOn == 'prompt' ? aiHelperCommandList[selection].prompt : '' );
						}
					} }
				>
					{ Object.keys( aiHelperCommandList ).map( value => {
						if ( value.slice( 0, 2 ) == '/-' ) {
							return (
								<optgroup label={ aiHelperCommandList[value].title } key={ value }></optgroup>
							);
						} else {
							return (
								<option value={ value } key={ value }>{ aiHelperCommandList[value].title }</option>
							);
						}
					} ) }
				</SelectControl>
			</PanelRow>
			{ aiHelperCommand == 'process_tone' && (
				<PanelRow>
					<SelectControl
						label={ __( 'Text tone:' ) }
						value={ aiHelperTextTone }
						options={ aiHelperTextToneOptions }
						onChange={ ( selection ) => setAiHelperTextTone( selection ) }
					/>
				</PanelRow>
			) }
			{ aiHelperCommand == 'process_translate' && (
				<PanelRow>
					<SelectControl
						label={ __( 'Language:' ) }
						value={ aiHelperTextLanguage }
						options={ aiHelperTextLanguageOptions }
						onChange={ ( selection ) => setAiHelperTextLanguage( selection ) }
					/>
				</PanelRow>
			) }
			<PanelRow>
				<SelectControl
					label={ __( 'Based on:' ) }
					value={ aiHelperBasedOn }
					options={ aiHelperBaseOptions }
					onChange={ ( selection ) => {
							setAiHelperBasedOn( selection );
							setAiHelperPrompt( selection == 'prompt' ? aiHelperCommandList[aiHelperCommand].prompt : '' );
							if ( selection == 'prompt' ) {
								setAiHelperHint( '' );
							}
						}
					}
				/>
			</PanelRow>
			{ false && (
				<PanelRow>
					<RadioControl
						label={ __( 'Based on:' ) }
						help={ __( 'What part of the content to send to AI for processing' ) }
						selected={ aiHelperBasedOn }
						options={ aiHelperBaseOptions }
						onChange={ ( value ) => {
								setAiHelperBasedOn( value );
								setAiHelperPrompt( value == 'prompt' ? aiHelperCommandList[aiHelperCommand].prompt : '' );
							}
						}
					/>
				</PanelRow>
			) }
			{ aiHelperBasedOn == 'prompt' && (
				<PanelRow>
					<TextareaControl
						label={ __( 'Prompt for AI:' ) }
						help={ __( 'Write a prompt with a brief description of what you want to get in response' ) }
						value={ aiHelperPrompt }
						onChange={ ( value ) => setAiHelperPrompt( value ) }
					/>
				</PanelRow>
			) }
			{ aiHelperBasedOn != 'prompt' && (
				<PanelRow>
					<TextareaControl
						label={ __( 'Additional Hint for AI:' ) }
						help={ __( 'Write a hint with additional requirements/instructions for the AI ​​response' ) }
						value={ aiHelperHint }
						onChange={ ( value ) => setAiHelperHint( value ) }
					/>
				</PanelRow>
			) }
			<PanelRow>
				<Button variant="primary" disabled={ busy } isBusy={ busy } className="trx_addons_ai_helper_button" onClick={() => getAiResponse()}>
					<Icon className="trx_addons_ai_helper_button_icon" icon={ busy ? 'update' : 'welcome-write-blog' } />
					<span className="trx_addons_ai_helper_button_text">{ __( 'Generate' ) }</span>
				</Button>
			</PanelRow>
		</PluginDocumentSettingPanel>
	);
};

registerPlugin( 'trx-addons-ai-helper-setting-panel', {
    render: TrxAddonsAiHelperSettingPanel,
    icon: 'rest-api',
} );
